/* FILE: mdsp.c                                 (J. Rogers      10/24/90)

This is a collection of C helper functions that manage the interface between
Symmetric Research's DSP32C board and xdetect.  All functions have been
written and compiled medium model.  The following functions are included:

display_bands ()            write band info to stream
do_chan_fft ()              takes fft of single waveform channel
combine_chan_bands ()       conbine into user-defined bands
update_activity_counter ()  the RSAM simulator
check_chan_spectra ()       check fft for spectral signal
dsp_get_spectral_channel () return spectral trigger channel
dsp_get_status ()           return char for spectral signal
dsp_spectral_detection () return flag for spectral_detection
dsp_spectral_detection_done ()     return flag for spectral recording done
dsp_set_MaxCalibrationTime () passes max calibration time to mdsp
dsp_set_spectral_status ()  passes whether spectral recording is ON or OFF to mdsp
dsp_set_fft_file_status ()  passes whether bands recording is ON or OFF to mdsp
dsp_set_bufs_to_avg ()      passes numbers of buffers to average to mdsp
dsp_set_band ()             passes bands definitions to mdsp
dsp_domain_check ()         main mdsp routine called from xdetect
dsp_initialize ()           initialize spectrum
dsp_continue_band_recording ()    continue to record bands thru events

EXTERNAL FUNCTIONS CALLED:

spectrum ()                          compute an FFT            (dspmath.lib)

HISTORY:
   Oct, 1990 - added MDETECT DSP modules to XDETECT



/*************************************************************************
                             INCLUDE FILES


*************************************************************************/
#include <stdio.h>
#include <io.h>
#include <malloc.h>
#include <math.h>
#include <time.h>
#include "mconst.h"
#include "mdemux.h"
#include "mtrigger.h"
#include "mdsp.h"
#include "mdt28xx.h"
#include "xdetect.h"
#include "mqueue.h"
#include <conio.h>
#include <dos.h>
#include "suds.h"
#include "dspmath.h"

/*************************************************************************
                                GLOBALS

*************************************************************************/
char end_string[] = "* * END OF DATA * *\x0D";
PRIVATE FLAG spectral_detection;             /* indicates freq. domain trig */
PRIVATE int spectral_channel;                /* chan with fft trigger */
PRIVATE int line_band_limits[NBANDS*2];       /* spectral lines defined here */
PRIVATE float freq_band_limits[NBANDS*2];       /* band definitions in Hz */
PRIVATE int bufs_to_avg;               /* 5.12 sec windows in avg */
PRIVATE char fft_file_name[24];        /* string for fft file name */
PRIVATE double max_calib_time;         /* length in seconds of A1VCO calib */
PRIVATE FLAG spectral_recording_enabled;
PRIVATE FLAG fft_file_enabled;
PRIVATE time_t ltime1, ltime2;
PRIVATE FLAG dsp_enabled;
PRIVATE float taper[NPTFFT];            /* window array for spectrum () */
PRIVATE char event_type;                /* either calibration or event */
PRIVATE int cal_line;                   /* 21.25 calibration line */
PRIVATE int rsam_ratio;                 /* RSAM ratio for incrementing ctr */
PRIVATE int rsam_thrsh;                 /* RSAM threshold for incr counter */
PRIVATE int low_line, high_line;        /* RSAM bands of interest */
PRIVATE unsigned int channel_activity_ctr[NCHAN]; /* number of events on ch */
extern Q_STATION far * st_get_station();

/* structure for fft suds data header tag */
struct fft_suds_header_struct
  {
  CHAR sync;                  /* always 'S' */
  CHAR machine;               /* '6' for pc */
  SH_INT id_struct;           /* id = 31 for header struct  */
  LG_INT len_struct;          /* length of header struct */
  LG_INT len_data;            /* not used = 0 */
  };
/* structure for fft data header - explains band definitions */
struct fft_header_struct
  {
  SH_INT nchan;               /* number of channels */
  SH_INT avg_bands;           /* number of bands */
  CHAR chan_id[NCHAN*6];      /* string of channel id codes */
  SH_INT num_bufs;            /* number of 5.12 sec buffers in avg */
  SH_INT pad;                 /* to allign to 4-byte boundries */
  FLOAT band_limits[NBANDS*2]; /* bands limits for fft bands */
  };

/* struct for fft data with seconds since 1970 */
  struct fft_data_struct
  {
  SH_INT nchan;              /* number of channels */
  SH_INT avg_bands;          /* number of bands */
  ST_TIME seconds;   /* seconds since 1/1/70 after fft average */
  SH_INT fft_channel_data[NCHAN*NBANDS];  /* fft data */
  };

/* declare suds structures */
  struct fft_suds_header_struct fft_suds_header;
  struct fft_header_struct fft_header;
  struct fft_data_struct fft_data;

/*=======================================================================*
 *                         display_bands                                    *
 *=======================================================================*/
/* Write band information to binary file and parallel port */
void display_bands ( b, nbands )
   int * b;         /* b -> band info */
   int nbands;

   {
   FILE * stream;
   static int start;
   int file_exist;
   static int buf_ctr;
   static long far avg_bands[NCHAN*NBANDS];
   int m,n,temp,i;
   struct tm *dos_time;
   char * fft_file_name_ptr;
   struct tm * date_ptr;
   long(ltime);
   char *suffix = ".fft";
   char *prefix = "c:\\data\\";
   time_t aclock;
   char buffer[20];
   static char out_string[200];
   char * out_ptr;
   char * end_ptr;
   union REGS regs;
   static unsigned int last_activity_ctr[NCHAN];

   /* structure for fft suds data header tag */
   struct fft_suds_data_struct
     {
     CHAR sync;                 /* always 'S' */
     CHAR machine;              /* '6' for pc */
     SH_INT id_struct;          /* id = 32 for fft data struct */
     LG_INT len_struct;         /* length of fft data struct */
     LG_INT len_data;           /* not used = 0 */
     };

   struct fft_suds_data_struct fft_suds_data;

   fft_file_name_ptr = fft_file_name;

   /* fill in suds data struct */
   fft_suds_data.sync = 'S';
   fft_suds_data.machine = '6';
   fft_suds_data.id_struct = 32;
   fft_suds_data.len_struct = (long)sizeof(fft_data);
   fft_suds_data.len_data = 0L;

   end_ptr = end_string;
   for( n = 0; n < NCHAN; n++ )
     for ( m = 0; m < NBANDS; m++ )
       if(start != 0)              /* throw away junk from 1st fft buffer */
	 avg_bands[m+NBANDS*n] += (long)b[m+NBANDS*n];
   buf_ctr++;
   if(start == 0) start++;

   if( buf_ctr % bufs_to_avg == 0 )
     {
     /* create file name for fft data yrmodyhr.fft */
     time(&aclock);
     dos_time = localtime(&aclock);
     i = sprintf(fft_file_name_ptr, "%s", prefix);
     i += sprintf(fft_file_name_ptr+i,"%d", dos_time->tm_year);
     if( dos_time->tm_mon < 9 )
       i += sprintf(fft_file_name_ptr+i,"%c", '0');
     i += sprintf(fft_file_name_ptr+i,"%d", dos_time->tm_mon+1);
     if( dos_time->tm_mday < 10 )
       i += sprintf(fft_file_name_ptr+i,"%c", '0');
     i += sprintf(fft_file_name_ptr+i,"%d", dos_time->tm_mday);
     if( dos_time->tm_hour < 10 )
       i += sprintf(fft_file_name_ptr+i,"%c", '0');
     i += sprintf(fft_file_name_ptr+i,"%d", dos_time->tm_hour);
     i += sprintf(fft_file_name_ptr+i,"%s", suffix);


     for( n = 0; n < NCHAN; n++)
       {
       out_ptr = out_string;
       strcpy(out_string,"@");
       if( n < 10 ) strcat(out_string, "0");
       sprintf(buffer, "%d", n);
       strcat(out_string, buffer);
       strcat(out_string," ");
       sprintf(buffer,"%d ", dos_time->tm_year);
       strcat(out_string, buffer);
       if( dos_time->tm_yday < 100 ) strcat(out_string, "0");
       sprintf(buffer,"%d ", dos_time->tm_yday+1);
       strcat(out_string, buffer);
       if( dos_time->tm_hour < 10 ) strcat(out_string, "0");
       sprintf(buffer,"%d", dos_time->tm_hour);
       strcat(out_string, buffer);
       strcat(out_string, ":" );
       if( dos_time->tm_min < 10 ) strcat(out_string, "0");
       sprintf(buffer,"%d ", dos_time->tm_min);
       strcat(out_string, buffer);
       for( m = 0; m < NBANDS; m++ )
         {
         sprintf(buffer, "%4d> ", (int)avg_bands[m+NBANDS*n]/bufs_to_avg);
         strcat(out_string, buffer);
         }
       sprintf(buffer,"%d ",channel_activity_ctr[n]-last_activity_ctr[n]);
       strcat(out_string, buffer);
       strcat(out_string, "@\x0D");

       /* check to see if printer LPT1 port is ready */
       regs.h.ah = 2;
       regs.h.al = 0;
       regs.h.dh = 0;
       regs.h.dl = 0;
       int86(0x17, &regs, &regs);

       if( regs.h.ah == 0x90 )
         while(*out_ptr) bdos( 0x5, *out_ptr++, 0 );
       }

     if( regs.h.ah == 0x90 )
       while(*end_ptr) bdos( 0x5, *end_ptr++, 0 );

     /* fill in fft_data structure and write out */
     fft_data.nchan = NCHAN;
     fft_data.avg_bands = NBANDS;
     fft_data.seconds = aclock;
     for ( n = 0; n < NCHAN; n++ )
        for ( m = 0; m < NBANDS; m++)
          {
          fft_data.fft_channel_data[m+n*NBANDS] = (int)avg_bands[m+NBANDS*n]/bufs_to_avg;
	  avg_bands[m+NBANDS*n] = 0;
          }

     file_exist = access(fft_file_name, 0);     /* see if fft file exists */

     stream = fopen(fft_file_name, "ab" );
     if(stream == NULL)
       {
       printf("could not open %s\n", fft_file_name);
       exit(0);
       }

     if( file_exist == -1 )                     /* add header if new fft */
       {
       fwrite(&fft_suds_header,sizeof(fft_suds_header),1,stream);
       fwrite(&fft_header,sizeof(fft_header),1,stream);
       }
     fwrite(&fft_suds_data,sizeof(fft_suds_data),1,stream);
     fwrite(&fft_data,sizeof(fft_data),1,stream);
     fclose(stream);
     /* reset counter to write data and update rsam activity counter */
     for ( n = 0; n < NCHAN; n++) last_activity_ctr[n] = channel_activity_ctr[n];
     buf_ctr = 0;
     }
   }


/*=======================================================================*
 *                                do_chan_fft                            *
 *=======================================================================*/
/* Take an fft of a single time channel to form fft array. The real time
series pointed to by p is converted to a complex time series. Then an fft is
taken on the complex time series and this is converted to an absolute amplitude
spectrum which is returned */

PRIVATE
FLAG do_chan_fft ( channel, npts, r )
   int channel;
   int npts;                    /* number of points of fft */
   int far *r;
   {
   int far * demux_ptr;       /* ptr to struct containing buffers */
   int j;
   static int far time_array[NPTFFT];
   static float waveform[NPTFFT];
   static float waveform_spectrum[NPTFFT];
   unsigned int dc_offset;

   /* fill the buffer with waveform data from a channel */
   demux_ptr = (dm_get_head_buffer())->data + channel*BUF_SIZ;   /* pointer to channel */
   for ( j = 0; j < BUF_SIZ; j++) time_array[j] = demux_ptr[j];
   demux_ptr = (dm_get_next_buffer())->data + channel*BUF_SIZ;
   for ( j = 0; j <  BUF_SIZ; j++) time_array[j+BUF_SIZ] = demux_ptr[j];

   dc_offset = dt_get_dc_offset();
   for (j = 0; j < npts; j++)           /* remove binary coding offset */
      time_array[j] -= dc_offset;
   for( j = 0; j < NPTFFT; j++) waveform[j] = (float)time_array[j];
   /* take fft and put in r array */
   spectrum(waveform,taper,NPTFFT,waveform_spectrum);
   for( j = 0; j < NPTFFT/2; j++) r[j] =(int)(waveform_spectrum[j]/NPTFFT);
   return TRUE;                          /* got the spectrum */
   }

/*=======================================================================*
 *                       combine_chan_bands                              *
 *=======================================================================*/
/* Combine fft results into nbands for a channel. The channel fft array
pointed to by r is reduced to NBANDS for convenience. Each npts/(2*NBANDS)
lines are averaged with the upper npts/2 lines ignored. They are redundant.
A pointer is returned to the reduced array.  */

PRIVATE
FLAG combine_chan_bands( r, b, npts, nbands, band_limits )
   int far * r;                 /* r -> chan fft */
   int far * b;                 /* b -> channel bands */
   int npts;                    /* number of points of fft */
   int nbands;                  /* number of bands */
   int band_limits[2*NBANDS]; /* line limits for bands */

   {
   int j, k, m;
   int lo_line, hi_line;        /* line limits for bands */
   int avg_line;                /* avg ampl of spectral line */
   long temp;

   /* initialize arays */
   for( j = 0; j < NBANDS; j++ ) b[j] = 0;

   m = 0;
   /* get average band info */
   for ( j = 0; j < 2*NBANDS; j +=2 )
      {
      b[m] = 0;
      lo_line = band_limits[j]; hi_line = band_limits[j+1];
      for (k = lo_line; k < hi_line; k++ )
          b[m] += r[k];
      if( hi_line != lo_line)
          b[m] /= hi_line-lo_line;    /* get avg amplitude in band */
      m++;                                 /* next band */
      }
   return TRUE;    /* return pointer to banded fft data */
   }

/*=======================================================================*
 *                      check_chan_calib                                 *
 *=======================================================================*/
/* Checks spectral lines against known cabib and returns an flag
if a calibration from A1VCO (energy at 21.25Hz) or other spectral signal
is found */

PRIVATE
FLAG check_chan_calib (r, b, channel )
 int far * r;         /* ptr to fft channel data */
 int far * b;         /* ptr to band channel data */
 int channel;
 {
 int j;                         /* counting indices */
 int avg_ampl = 1;                 /* avg amplitude band */
 int cal_ampl;                     /* signal amplitudes */
 static int start;

  if ( start < NCHAN*10 )         /* hold off till start-up transient dies */
    start++;

 avg_ampl = 1;                  /* initialize */
 for ( j = 0; j < NBANDS; j++ )   /* process every band */
    {
    avg_ampl += *b;
    b++;
    }
 avg_ampl /= NBANDS;                 /* get average spectral line ampl */
 avg_ampl = ( avg_ampl < 1 ) ? 1 : avg_ampl;

 /* check for calib frequency signal */
 cal_ampl = r[cal_line-1] + r[cal_line] + r[cal_line+1];
 if ( (cal_ampl/avg_ampl > CALTHR) && (cal_ampl > CAL_ABS_THR) && (start > NCHAN*9) )
   {
   event_type = DSP_CALIBRATION;
   return(TRUE);
   }
 else
   return(FALSE);
 }

/*=======================================================================*
 *                      check_chan_spectra                               *
 *=======================================================================*/
/* Checks spectral lines for a band of frequencies and returns a flag
if energy in this band is present */

PRIVATE
FLAG check_chan_spectra (r, channel )
 int far * r;         /* ptr to fft channel data */
 int channel;
  {
  int j;
  long new_spectral_value;
  static long spectral_sta[NCHAN];
  static long spectral_lta[NCHAN];       /* spectral lta array */
  static int start = 0;
  static time_t last_channel_trigger[NCHAN];
  static FLAG current_triggers[NCHAN];
  static unsigned int trigger_ctr;


  /* get new spectral sta and lta of sum of bands of interest */
  new_spectral_value = 0;
  for( j = LOW_BAND_EDGE; j < HIGH_BAND_EDGE; j++)
    new_spectral_value += r[j];
  spectral_sta[channel] += (new_spectral_value-spectral_sta[channel])/STA_TIME;
  spectral_lta[channel] += (new_spectral_value-spectral_lta[channel])/LTA_TIME;
  spectral_lta[channel] = (spectral_lta[channel] < 1L) ? 1L : spectral_lta[channel];
  if ( start < NCHAN*10 )         /* hold off till start-up transient dies */
    {
    start++;
    return(FALSE);
    }

  /* throw out old triggers */
  time(&ltime1);
  for ( j = 0; j < NCHAN; j++ )
    if(ltime1 - last_channel_trigger[j] > TIME_WINDOW && current_triggers[j] == TRUE)
      {
      trigger_ctr--;
      current_triggers[j] = FALSE;
      }

/* debug
if(channel == 0)
  {
  for( j = 0; j < NCHAN; j++) printf("%d ",current_triggers[j]);
  printf("\n");
  }      */

  /*check if spectral signal of interest */
  if (t_get_channel_trigger_state (channel) && (float)spectral_sta[channel]/(float)spectral_lta[channel] >= SPECTRAL_THR )
    {
    time(&ltime1);
    last_channel_trigger[channel] = ltime1;
    if(ltime1 - last_channel_trigger[channel] < TIME_WINDOW && current_triggers[channel] == FALSE)
      {
      trigger_ctr++;
      current_triggers[channel] = TRUE;
      }
    if( trigger_ctr >= TRIGGERS_IN_WINDOW )
      {
      trigger_ctr = 0;
      for ( j = 0; j < NCHAN; j++ ) current_triggers[j] = FALSE;
      event_type = DSP_EVENT;
      return(TRUE);
      }
    return(FALSE);
    }
  else
    return(FALSE);
  }


/*=======================================================================*
 *                      update_activity_counter                          *
 *=======================================================================*/
/* Updates the activity counter for each channel. Channel activity is
considered significant if avg[time]/avg[time-2] > 2 and avg[time] > threshold
If these conditions are met then avg[time-1] is set to a large number */

PRIVATE
void update_activity_counter (r, channel )
 int far * r;         /* ptr to fft channel data */
 int channel;
  {
  static unsigned int channel_activity[NCHAN*3];
  int j;
  unsigned index;
  unsigned static bank;
  unsigned int avg_spectral_value;

  /* set activity indices */
  index = channel + NCHAN*bank;
  if(channel == NCHAN-1) bank = (bank+1) % 3;
  channel_activity[(index+NCHAN)%(3*NCHAN)] = (channel_activity[(index+NCHAN)%(3*NCHAN)] < 1) ? 1 : channel_activity[(index+NCHAN)%(3*NCHAN)];

  /* update spectral activity of avg of bands of interest */
  avg_spectral_value = 0;
  for( j = low_line; j < high_line; j++)
    avg_spectral_value += r[j];
  avg_spectral_value /= (high_line - low_line);
  channel_activity[index] = avg_spectral_value;

  if (channel_activity[index]/channel_activity[(index+NCHAN)%(3*NCHAN)] >= rsam_ratio
    && channel_activity[index] > rsam_thrsh )
    {
    channel_activity_ctr[channel]++;
    channel_activity[(index+2*NCHAN)%(3*NCHAN)] = 20000;
    }
  }

/*=======================================================================*

/*=======================================================================*
 *                      continue_spectral_recording                      *
 *=======================================================================*/
/* Checks the spectral trigger to determine if the recording should be
continued. The duration of recording is based on the spectral STA/LTA
ratio */

PRIVATE
FLAG continue_spectral_recording (r, channel )
 int far * r;         /* ptr to fft channel data */
 int channel;
  {
  int j;
  long new_spectral_value;
  static long spectral_sta[NCHAN];
  static long spectral_lta[NCHAN];       /* spectral lta array */
  static time_t last_channel_trigger[NCHAN];
  static FLAG current_triggers[NCHAN];
  static unsigned int trigger_ctr;
  }

/*=======================================================================*
 *                      dsp_get_spectral_channel                            *
 *=======================================================================*/

PUBLIC
unsigned int dsp_get_spectral_channel ()
{
   return(spectral_channel);
}

/*=======================================================================*
 *                           dsp_get_dsp_status                          *
 *=======================================================================*/
/* Get dsp_enabled.                                                      */

PUBLIC
FLAG dsp_get_dsp_status ()
{
   return dsp_enabled;
}


/*=======================================================================*
 *                              dsp_get_status                           *
 *=======================================================================*/

PUBLIC
char dsp_get_status ()
{
   return((spectral_detection) ? DSP_CALIBRATION : FALSE);
}

/*=======================================================================*
 *                         dsp_spectral_detection                        *
 *=======================================================================*/

PUBLIC
FLAG dsp_spectral_detection ()
{
   return (spectral_detection);
}

/*=======================================================================*
 *                           dsp_spectral_detection_done                 *
 *=======================================================================*/

PUBLIC
FLAG dsp_spectral_detection_done ()
  {
  static int i;
  time(&ltime2);
  switch(event_type)
    {
    case DSP_EVENT:
      if ( i < DSP_EVENT_RECORDING_TIME)
        {
        i++;
        return(FALSE);
        }
      else
        {
        i = 0;
        spectral_detection = FALSE;
        return(TRUE);
        }
      break;
    case DSP_CALIBRATION:
      if((ltime2-ltime1) <= (long)max_calib_time )
        return(FALSE);
      else
        {
        spectral_detection = FALSE;
        return(TRUE);
        }
      break;
    default:
      if(Debug_enabled)
        printf("Error in spectral detector\n");
      exit(1);
    }
  }

/*=======================================================================*
 *                          dsp_set_MaxCalibrationTime                   *
 *=======================================================================*/

PUBLIC
void dsp_set_MaxCalibrationTime (time)
double time;
{
   max_calib_time = time;
}

/*=======================================================================*
 *                       dsp_set_spectral_status                      *
 *=======================================================================*/
/* Set spectral_enabled flag                                       */

PUBLIC
void dsp_set_spectral_status (spectral_status)
FLAG spectral_status;
{
   spectral_recording_enabled = spectral_status;
}

/*=======================================================================*
 *                           dsp_set_dsp_status                          *
 *=======================================================================*/
/* Set dsp_enabled flag.                                                 */

PUBLIC
void dsp_set_dsp_status (dsp_status)
FLAG dsp_status;
{
   dsp_enabled = dsp_status;
}

/*=======================================================================*
 *                         dsp_set_fft_file_status                           *
 *=======================================================================*/
/* Set fft_file_enabled flag.                                                 */

PUBLIC
void dsp_set_fft_file_status (fft_file_status)
FLAG fft_file_status;
{
   fft_file_enabled = fft_file_status;
}

/*=======================================================================*
 *                         dsp_set_bufs_to_avg                           *
 *=======================================================================*/
/* Set buffers to average.                                               */

PUBLIC
void dsp_set_bufs_to_avg (number)
int number;
{
   bufs_to_avg = number;
}

/*=======================================================================*
 *                         dsp_set_band                                  *
 *=======================================================================*/
/* Set band limits.                                               */

PUBLIC
void dsp_set_band (low, high)
float low, high;
{
static int i;
freq_band_limits[i++] = low; freq_band_limits[i++] = high;
}

/*=======================================================================*
 *                         dsp_domain_check                              *
 *=======================================================================*/
/* Analyze the current buffer for characteristic spectral lines and      */
/* return flag to signal main() that an interesting signal is occurring. */

PUBLIC
FLAG dsp_domain_check ()
   {
   int far * data_ptr;
   int i,j;
   int far * r1;
   int far * b1;
   static int b[NCHAN*NBANDS];

   if(spectral_recording_enabled || fft_file_enabled)
      {
      r1 = (int far *) _fmalloc(NPTFFT*sizeof(int)/2);
      if( r1 == NULL )
        {
        if(Debug_enabled) printf("could not allocate buffer space for r1 in mdsp.c\n");
        exit(1);
        }
      b1 = (int far *) _fmalloc(NBANDS*sizeof(int));
      if( b1 == NULL )
        {
        if(Debug_enabled) printf("could not allocate buffer space for b1 in mdsp.c\n");
        exit(1);
        }
      for ( i = 0; i < NCHAN ; i++ )     /* check for A1VCO characteristic signals */
         {
         if (do_chan_fft(i, NPTFFT, r1));
         combine_chan_bands(r1, b1, NPTFFT, NBANDS, line_band_limits);
         update_activity_counter(r1, i);
         for ( j = 0; j < NBANDS; j++ )
           b[i*NBANDS+j] = b1[j];
         if( check_chan_spectra(r1, i) || check_chan_calib(r1, b1, i) )
            {
            spectral_detection = TRUE;           /* do if fft trig */
            spectral_channel = i;
            break;
            }
         else spectral_detection = FALSE;               /* no fft trig */
         }
      s_display_triggers ();
      if(fft_file_enabled) display_bands(b, NBANDS);
      _ffree(r1);
      _ffree(b1);
      }
   if (spectral_detection) time(&ltime1);
   return(spectral_detection);
   }


/*=======================================================================*
 *                         dsp_continue_band_recording                   *
 *=======================================================================*/
/* Continue to record fft band info into .fft file                       */

PUBLIC
void dsp_continue_band_recording ()
   {
   int far * data_ptr;
   int i,j;
   int far * r1;
   int far * b1;
   static int b[NCHAN*NBANDS];

   if(fft_file_enabled)
      {
      r1 = (int far *) _fmalloc(NPTFFT*sizeof(int)/2);
      if( r1 == NULL )
        {
        if(Debug_enabled) printf("could not allocate buffer space for r1 in mdsp.c\n");
        exit(1);
        }
      b1 = (int far *) _fmalloc(NBANDS*sizeof(int));
      if( b1 == NULL )
        {
        if(Debug_enabled) printf("could not allocate buffer space for b1 in mdsp.c\n");
        exit(1);
        }
      for ( i = 0; i < NCHAN ; i++ )     /* check for A1VCO characteristic signals */
         {
         if (do_chan_fft(i, NPTFFT, r1));
         combine_chan_bands(r1, b1, NPTFFT, NBANDS, line_band_limits);
         for ( j = 0; j < NBANDS; j++ )
           b[i*NBANDS+j] = b1[j];
         }
      display_bands(b, NBANDS);
      _ffree(r1);
      _ffree(b1);
      }
   }


/*=======================================================================*
 *                        dsp_initialize_params                          *
 *=======================================================================*/
/* Initialize key parameters.                                            */

PUBLIC
void dsp_initialize_params ()
{
   dsp_enabled = DSP_ENABLED;
   spectral_recording_enabled = CALIBRATION_ENABLED;
   fft_file_enabled = FFT_FILE_ENABLED;
   spectral_detection = FALSE;
}

/*=======================================================================*
 *                         dsp_initialize                                *
 *=======================================================================*/
/* initialize arrays and variables used in DSP package */
PUBLIC
void dsp_initialize()
  {
  FILE * stream;
  int i,k,l;
  int j = 0;
  struct tm * date_ptr;
  long ltime;
  char info_file_name[30];
  char * info_file_name_ptr;
  char *suffix_asc = ".inf";
  char station_id_list[NCHAN*6];
  char *fft_file_name_ptr;
  char *suffix = ".fft";
  char *path_name_ptr;
  float low_cut, high_cut;        /* RSAM freq. bands to avg */

  /* check channel block size */
  if(dt_get_channel_size() !=  BUF_SIZ)
    {
    printf("Exiting program\n");
    printf("Change input file ChannelBlockSize from %d to 256!\n", dt_get_channel_size());
    exit(0);
    }

  /* read in RSAM values */
  if (( stream = fopen("rsam.fft", "r+t")) == NULL)
    {
    printf("Could not find rsam input file\n");
    exit(0);
    }
  fscanf(stream, "%f %f", &low_cut, &high_cut);
  fscanf(stream, "%d %d", &rsam_ratio,&rsam_thrsh);
  /* calculate equivalent line numbers given sample rate etc */
  low_line = (int)(low_cut*NPTFFT/dt_get_digitization_rate())+1;
  high_line = (int)(high_cut*NPTFFT/dt_get_digitization_rate())+1;

   /* get station id's */
   for( i = 0; i < NCHAN; i++ )
     for ( l = 0; l < 4; l++)
       {
       station_id_list[i*5+l] = st_get_station(i)->info.sc_name.st_name[l];
       station_id_list[i*5+4] = ' ';
       }
   station_id_list[NCHAN*5-1] = '\0';

  /* initialize string pointers */
  fft_file_name_ptr = fft_file_name;
  path_name_ptr = (char *)f_get_pathname();
  info_file_name_ptr = info_file_name;

  /* create file name for fft data */
  time(&ltime);
  date_ptr = localtime(&ltime);
  i = sprintf(fft_file_name_ptr, "%s", path_name_ptr);
  i += sprintf(fft_file_name_ptr+i,"%d", date_ptr->tm_year);
  if( date_ptr->tm_mon < 9 )
    i += sprintf(fft_file_name_ptr+i,"%c", '0');
  i += sprintf(fft_file_name_ptr+i,"%d", date_ptr->tm_mon+1);
  if( date_ptr->tm_mday < 10 )
    i += sprintf(fft_file_name_ptr+i,"%c", '0');
  i += sprintf(fft_file_name_ptr+i,"%d", date_ptr->tm_mday);
  if( date_ptr->tm_hour < 10 )
    i += sprintf(fft_file_name_ptr+i,"%c", '0');
  i += sprintf(fft_file_name_ptr+i,"%d", date_ptr->tm_hour);
  i += sprintf(fft_file_name_ptr+i,"%s", suffix);

  /* create file name for ascii file */
  i = sprintf(info_file_name_ptr, "%s", path_name_ptr);
  i += sprintf(info_file_name_ptr+i,"%d", date_ptr->tm_year);
  if( date_ptr->tm_mon < 9 )
    i += sprintf(info_file_name_ptr+i,"%c", '0');
  i += sprintf(info_file_name_ptr+i,"%d", date_ptr->tm_mon+1);
  if( date_ptr->tm_mday < 10 )
    i += sprintf(info_file_name_ptr+i,"%c", '0');
  i += sprintf(info_file_name_ptr+i,"%d", date_ptr->tm_mday);
  if( date_ptr->tm_hour < 10 )
    i += sprintf(info_file_name_ptr+i,"%c", '0');
  i += sprintf(info_file_name_ptr+i,"%d", date_ptr->tm_hour);
  i += sprintf(info_file_name_ptr+i,"%s", suffix_asc);

  /* convert freq in Hz to band numbers */
  for ( i = 0; i < NBANDS*2; i++)
    line_band_limits[i] = (int)(freq_band_limits[i]*NPTFFT/dt_get_digitization_rate())+1;
  fclose(stream);

  /* find 21.25 Hz calibration line number */
  cal_line = (int)(((float)NPTFFT/dt_get_digitization_rate())*21.25);

 if(Debug_enabled)
    {
    printf("FFT data located in %s\n",fft_file_name);
    if(fft_file_enabled) printf("FFT monitoring file enabled and ");
    else printf("FFT monitoring file disabled and");
    if(spectral_recording_enabled) printf("spectral recording enabled\n");
    else printf("spectral recording disabled.\n");
    }

  /* define rectangular taper array for windowing fft array */
   for ( i = 0; i < NPTFFT; i++ ) taper[i] = 1.0;

  /* fill in suds header struct */
  fft_suds_header.sync = 'S';
  fft_suds_header.machine = '6';
  fft_suds_header.id_struct = 31;
  fft_suds_header.len_struct = (long)sizeof(fft_header);
  fft_suds_header.len_data = 0L;

  /* fill in fft_header structure */
  strcpy(fft_header.chan_id, station_id_list);
  fft_header.nchan = NCHAN;
  fft_header.avg_bands = NBANDS;
  fft_header.num_bufs = bufs_to_avg;
  fft_header.pad = 0;
  for ( i = 0; i < NBANDS*2; i++ )
     fft_header.band_limits[i] = freq_band_limits[i];

  /* initialize band array */
  for ( i = 0; i < NCHAN; i++ )
     for ( j = 0; j < NBANDS; j++)
       fft_data.fft_channel_data[j+i*NBANDS] = 0;

  /* write binary file */
  stream = fopen(fft_file_name, "ab" );
  if (stream == NULL)
    {
    printf("\nCould not open %s file\n",fft_file_name);
    exit(0);
    }
  fwrite(&fft_suds_header,sizeof(fft_suds_header),1,stream);
  fwrite(&fft_header,sizeof(fft_header),1,stream);
  fclose(stream);

  /* write out ascii info file */
  stream = fopen(info_file_name, "w" );
  if (stream == NULL)
    {
    printf("\nCould not open %s file\n",info_file_name);
    exit(0);
    }
  fprintf(stream,"This .inf file is an information file which explains the format\n");
  fprintf(stream,"of the corresponding .fft file. The first 130 bytes are:\n");
  fprintf(stream,"Number of 512-point buffers in the average: %d\n", bufs_to_avg);
  fprintf(stream,"0 -- B0 from %5.1f to %5.1f Hz\n",freq_band_limits[0],freq_band_limits[1]);
  fprintf(stream,"1 -- B1 from %5.1f to %5.1f Hz\n",freq_band_limits[2],freq_band_limits[3]);
  fprintf(stream,"2 -- B2 from %5.1f to %5.1f Hz\n",freq_band_limits[4],freq_band_limits[5]);
  fprintf(stream,"3 -- B3 from %5.1f to %5.1f Hz\n",freq_band_limits[6],freq_band_limits[7]);
  fprintf(stream,"4 -- B4 from %5.1f to %5.1f Hz\n",freq_band_limits[8],freq_band_limits[9]);
  fprintf(stream,"5 -- B5 from %5.1f to %5.1f Hz\n",freq_band_limits[10],freq_band_limits[11]);
  fprintf(stream,"6 -- B6 from %5.1f to %5.1f Hz\n",freq_band_limits[12],freq_band_limits[13]);
  fprintf(stream,"7 -- B7 from %5.1f to %5.1f Hz\n",freq_band_limits[14],freq_band_limits[15]);
  fprintf(stream,"8 -- B8 from %5.1f to %5.1f Hz\n",freq_band_limits[16],freq_band_limits[17]);
  fprintf(stream,"9 -- B9 from %5.1f to %5.1f Hz\n",freq_band_limits[18],freq_band_limits[19]);
  fprintf(stream,"10 -- B10 from %5.1f to %5.1f Hz\n",freq_band_limits[20],freq_band_limits[21]);
  fprintf(stream,"11 -- B11 from %5.1f to %5.1f Hz\n",freq_band_limits[22],freq_band_limits[23]);
  fprintf(stream,"12 -- B12 from %5.1f to %5.1f Hz\n",freq_band_limits[24],freq_band_limits[25]);
  fprintf(stream,"13 -- B13 from %5.1f to %5.1f Hz\n",freq_band_limits[26],freq_band_limits[27]);
  fprintf(stream,"14 -- B14 from %5.1f to %5.1f Hz\n",freq_band_limits[28],freq_band_limits[29]);
  fprintf(stream,"15 -- B15 from %5.1f to %5.1f Hz\n",freq_band_limits[30],freq_band_limits[31]);
  fclose(stream);
  }

